//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

Item {
    id : progressBar

    property alias progress : bar.progress
    property bool  animated : false

    implicitHeight : 10
    implicitWidth  : 100

    Image {
        id : overlaySrc

        source  : "qrc:/ui/images/progressbar-overlay.png"
        visible : false

        layer.enabled  : true
        layer.wrapMode : ShaderEffectSource.Repeat
        layer.smooth   : true
    }

    ShaderEffect {
        id : bar

        property real progress

        property real offset : 0.0

        readonly property color backgroundColor :
            UiColors.getColor(UiColors.ControlGrey20)
        readonly property color foregroundColor :
            UiColors.getColor(UiColors.ButtonNormalBlue)

        readonly property Image overlay      : overlaySrc
        readonly property real  overlayScale :
            Math.max(1, bar.width) / overlaySrc.width

        readonly property bool useOverlay : progressBar.animated

        anchors.fill : parent

        Timer {
            id : animationTimer

            interval : 35
            running  : progressBar.visible && progressBar.animated
            repeat   : true

            onRunningChanged : bar.offset = 0.0;
            onTriggered      : bar.offset += (1.0 / Math.max(1, bar.width))
        }

        Behavior on progress {
            SmoothedAnimation { velocity : 3 }
        }

        vertexShader : "
            attribute mediump vec4 qt_Vertex;
            attribute mediump vec2 qt_MultiTexCoord0;

            uniform mediump mat4 qt_Matrix;

            uniform mediump float overlayScale;
            uniform mediump float offset;

            varying mediump vec2 v_TexCoord;
            varying mediump vec2 v_OverlayCoord;

            void main() {
                gl_Position    = qt_Matrix * qt_Vertex;
                v_TexCoord     = qt_MultiTexCoord0;
                v_OverlayCoord = (qt_MultiTexCoord0 - vec2(offset, 0.0)) *
                    vec2(overlayScale, 1.0);
            }"

        fragmentShader : "
            uniform lowp float qt_Opacity;

            uniform mediump float progress;

            uniform lowp vec4 foregroundColor;
            uniform lowp vec4 backgroundColor;

            uniform sampler2D overlay;
            uniform bool      useOverlay;

            varying mediump vec2 v_TexCoord;
            varying mediump vec2 v_OverlayCoord;

            void main() {
                if (useOverlay) {
                    lowp vec4 overlayCol = texture2D(overlay, v_OverlayCoord);
                    gl_FragColor = mix(
                        mix(foregroundColor, overlayCol, overlayCol.a),
                        backgroundColor, step(progress, v_TexCoord.x));
                } else {
                    gl_FragColor = mix(
                        foregroundColor, backgroundColor,
                        step(progress, v_TexCoord.x));
                }
                gl_FragColor *= qt_Opacity;
            }"
    }
}
